/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Workfile: OS_Specific.cpp $
   Last Modification:
    $Author: $
    $Modtime: $
    $Revision: $
   
   Targets:
     Win32/ANSI   : no
     Win32/Unicode: no
     WinCE 5      : yes
     WinCE 6      : yes
 
   Description:
     Windows CE OS Abstraction Layer implementation.
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     1         20.07.2010  SS       initial version
 
     
**************************************************************************************/

#include "OS_Includes.h"
#include "OS_Dependent.h"
#include "cifXHWFunctions.h"
#include "DevStruct.h"


/* global vars to check keybord state -> ReadInput()/ */
int    g_iHit;
bool   g_fWaitForThread = false;
HANDLE gHandle          = 0;

static bool           s_fUsePerfCounters = false;
static LARGE_INTEGER  s_liPerfFrequency  = {0};


/*****************************************************************************/
/*! Windows CE Interrupt service thread
*   \param pvParam  Pointer to CE Driver device instance
*   \return 0                                                                */
/*****************************************************************************/
static DWORD WINAPI IST(void* pvParam)
{
  PCIFXCEDRV_DEVICEINSTANCE ptDrvDevInst = (PCIFXCEDRV_DEVICEINSTANCE)pvParam;
  
  while(1)
  {
    DWORD dwWaitResult = WaitForSingleObject(ptDrvDevInst->hIrqEvent, INFINITE);

    if(ptDrvDevInst->fStopIst)
      break;

    int iRet = cifXTKitISRHandler(&ptDrvDevInst->tDevInstance, 0);

    switch(iRet)
    {
    case CIFX_TKIT_IRQ_OTHERDEVICE:
    case CIFX_TKIT_IRQ_HANDLED:
    default:
      break;

    case CIFX_TKIT_IRQ_DSR_REQUESTED:
      cifXTKitDSRHandler(&ptDrvDevInst->tDevInstance);
      break;
    }

    InterruptDone(ptDrvDevInst->tIsrInfo.dwSysintr);
  }

  return 0;
}


/*****************************************************************************/
/*! Memory allocation function
*   \param ulSize    Length of memory to allocate
*   \return Pointer to allocated memory                                      */
/*****************************************************************************/
void* OS_Memalloc(uint32_t ulSize)
{
  return LocalAlloc(LPTR, ulSize);
}

/*****************************************************************************/
/*! Memory freeing function
*   \param pvMem Memory block to free                                        */
/*****************************************************************************/
void OS_Memfree(void* pvMem)
{
  LocalFree(pvMem);
}

/*****************************************************************************/
/*! Memory setting
*   \param pvMem     Memory block
*   \param bFill     Byte to use for memory initialization
*   \param ulSize    Memory size for initialization)                         */
/*****************************************************************************/
void OS_Memset(void* pvMem, uint8_t bFill, uint32_t ulSize)
{
  memset(pvMem, bFill, ulSize);
}

/*****************************************************************************/
/*! Copy memory from one block to another
*   \param pvDest    Destination memory block
*   \param pvSrc     Source memory block
*   \param ulSize    Copy size in bytes                                      */
/*****************************************************************************/
void OS_Memcpy(void* pvDest, void* pvSrc, uint32_t ulSize)
{
  memcpy(pvDest, pvSrc, ulSize);
}

/*****************************************************************************/
/*! Compare two memory blocks
*   \param pvBuf1    First memory block
*   \param pvBuf2    Second memory block
*   \param ulSize    Compare size in bytes
*   \return 0 if both buffers are equal                                      */
/*****************************************************************************/
int OS_Memcmp(void* pvBuf1, void* pvBuf2, uint32_t ulSize)
{
  return memcmp(pvBuf1, pvBuf2, ulSize);
}

/*****************************************************************************/
/*! Move memory
*   \param pvDest    Destination memory
*   \param pvSrc     Source memory
*   \param ulSize    Size in byte to move                                    */
/*****************************************************************************/
void OS_Memmove(void* pvDest, void* pvSrc, uint32_t ulSize)
{
  memmove(pvDest, pvSrc, ulSize);
}

/*****************************************************************************/
/*! Compare two ASCII string
*   \param pszBuf1   First buffer
*   \param pszBuf2   Second buffer
*   \return 0 if strings are equal                                           */
/*****************************************************************************/
int OS_Strcmp(const char* pszBuf1, const char* pszBuf2)
{
  return strcmp(pszBuf1, pszBuf2);
}

/*****************************************************************************/
/*! Compare characters of two strings without regard to case
*   \param pszBuf1   First buffer
*   \param pszBuf2   Second buffer
*   \param ulLen     Number of characters to compare
*   \return 0 if strings are equal                                           */
/*****************************************************************************/
int OS_Strnicmp(const char* pszBuf1, const char* pszBuf2, uint32_t ulLen)
{
  return _strnicmp(pszBuf1, pszBuf2, ulLen);
}  

/*****************************************************************************/
/*! Query the length of an ASCII string
*   \param szText    ASCII string
*   \return character count of szText                                        */
/*****************************************************************************/
int OS_Strlen(const char* szText)
{
  return (int)strlen(szText);
}

/*****************************************************************************/
/*! Copies one string to another monitoring the maximum length of the target
*   buffer.
*   \param szDest    Destination buffer
*   \param szSource  Source buffer
*   \param ulLength  Maximum length to copy
*   \return pointer to szDest                                                */
/*****************************************************************************/
char* OS_Strncpy(char* szDest, const char* szSource, uint32_t ulLength)
{
  return strncpy(szDest, szSource, ulLength);
}

/*****************************************************************************/
/*! Opens a file in binary mode
*   \param szFile     Full file name of the file to open
*   \param pulFileLen Returned length of the opened file
*   \return handle to the file, NULL mean file could not be opened           */
/*****************************************************************************/
void* OS_FileOpen(char* szFile, uint32_t* pulFileLen)
{
  FILE* hFile = fopen(szFile, "rb");
  void* pvRet = NULL;

  if(hFile != NULL)
  {
    long lFileSize;

    if( (0  != fseek(hFile, 0, SEEK_END))  ||
        (-1 == (lFileSize = ftell(hFile))) )
    {
      /* error seeking end of file */
      fclose(hFile);
      hFile = NULL;

    } else
    {
      *pulFileLen = (uint32_t)lFileSize;
      int iTemp = fseek(hFile, 0, SEEK_SET);
      pvRet = hFile;
    }
  }
  return pvRet;
}

/*****************************************************************************/
/*! Closes a previously opened file
*   \param pvFile Handle to the file being closed                            */
/*****************************************************************************/
void OS_FileClose(void* pvFile)
{
  FILE* hFile = (FILE*)pvFile;
  fclose(hFile);
}

/*****************************************************************************/
/*! Read a specific amount of bytes from the file
*   \param pvFile   Handle to the file being read from
*   \param ulOffset Offset inside the file, where read starts at
*   \param ulSize   Size in bytes to be read
*   \param pvBuffer Buffer to place read bytes in
*   \return number of bytes actually read from file                          */
/*****************************************************************************/
uint32_t OS_FileRead(void* pvFile, uint32_t ulOffset, uint32_t ulSize, void* pvBuffer)
{
  FILE*    hFile = (FILE*)pvFile;
  uint32_t ulRet = 0;

  if(0 == (fseek(hFile, ulOffset, SEEK_SET)))
  {
    ulRet = fread(pvBuffer, 1, ulSize, hFile);
  }

  return ulRet;
}

/*****************************************************************************/
/*! Sleep for a specific time
*   \param ulSleepTimeMs  Time in ms to sleep for                            */
/*****************************************************************************/
void OS_Sleep(uint32_t ulSleepTimeMs)
{
  Sleep(ulSleepTimeMs);
}

/*****************************************************************************/
/*! Create an interrupt safe locking mechanism (Spinlock/critical section)
*   \return handle to the locking object                                     */
/*****************************************************************************/
void* OS_CreateLock(void)
{
  LPCRITICAL_SECTION ptCritSec = (LPCRITICAL_SECTION)OS_Memalloc(sizeof(*ptCritSec));

  InitializeCriticalSection(ptCritSec);

  return ptCritSec;
}

/*****************************************************************************/
/*! Enter a critical section/spinlock
*   \param pvLock Handle to the locking object                               */
/*****************************************************************************/
void OS_EnterLock(void* pvLock)
{
  LPCRITICAL_SECTION ptCritSec = (LPCRITICAL_SECTION)pvLock;
  EnterCriticalSection(ptCritSec);
}

/*****************************************************************************/
/*! Leave a critical section/spinlock
*   \param pvLock Handle to the locking object                               */
/*****************************************************************************/
void OS_LeaveLock(void* pvLock)
{
  LPCRITICAL_SECTION ptCritSec = (LPCRITICAL_SECTION)pvLock;
  LeaveCriticalSection(ptCritSec);
}

/*****************************************************************************/
/*! Delete a critical section/spinlock object
*   \param pvLock Handle to the locking object being deleted                 */
/*****************************************************************************/
void OS_DeleteLock(void* pvLock)
{
  LPCRITICAL_SECTION ptCritSec = (LPCRITICAL_SECTION)pvLock;

  DeleteCriticalSection(ptCritSec);
  OS_Memfree(ptCritSec);
}


/*****************************************************************************/
/*! Retrieves the last-error code value.
*   \return Error code value                                                 */
/*****************************************************************************/
int OS_GetLastError(void)
{
 return GetLastError();
}

/*****************************************************************************/
/*! Create an Mutex object for locking code sections 
*   \return handle to the mutex object                                       */
/*****************************************************************************/
void* OS_CreateMutex(void)
{
  return CreateMutex( NULL,     /* LPSECURITY_ATTRIBUTES lpMutexAttributes*/
                      FALSE,    /* BOOL bInitialOwner,*/
                      NULL);    /* LPCTSTR lpName */
}

/*****************************************************************************/
/*! Wait for mutex
*   \param pvMutex    Handle to the Mutex locking object
*   \param ulTimeout  Wait timeout                                           
*   \return !=0 on succes                                                    */
/*****************************************************************************/
int OS_WaitMutex(void* pvMutex, uint32_t ulTimeout)
{
  return (WaitForSingleObject((HANDLE)pvMutex, ulTimeout) == WAIT_OBJECT_0) ? 1 : 0;
}

/*****************************************************************************/
/*! Release a mutex section section
*   \param pvMutex Handle to the locking object                              */
/*****************************************************************************/
void OS_ReleaseMutex(void* pvMutex)
{
  ReleaseMutex( (HANDLE) pvMutex);
}

/*****************************************************************************/
/*! Delete a Mutex object
*   \param pvMutex Handle to the mutex object being deleted                  */
/*****************************************************************************/
void OS_DeleteMutex(void* pvMutex)
{
  CloseHandle( (HANDLE) pvMutex);
}

/*****************************************************************************/
/*! Retrieve a counter based on millisecond used for timeout monitoring
*   \return Current counter value (resolution of this value will influence
*           timeout monitoring in driver/toolkit functions(                  */
/*****************************************************************************/
uint32_t OS_GetMilliSecCounter(void)
{
  uint32_t ulRet;

  if(s_fUsePerfCounters)
  {
    LARGE_INTEGER liTime = {0};

    QueryPerformanceCounter(&liTime);

    liTime.QuadPart /= s_liPerfFrequency.QuadPart;

    ulRet = liTime.LowPart;

  } else
  {
    ulRet = GetTickCount();
  }

  return ulRet;
}



/*****************************************************************************/
/*! Create an auto reset event
*   \return handle to the created event                                      */
/*****************************************************************************/
void* OS_CreateEvent(void)
{
  return (void*)CreateEvent(NULL, FALSE, FALSE, NULL);
}

/*****************************************************************************/
/*! Set an event
*   \param pvEvent Handle to event being signalled                           */
/*****************************************************************************/
void OS_SetEvent(void* pvEvent)
{
  SetEvent((HANDLE)pvEvent);
}

/*****************************************************************************/
/*! Reset an event
*   \param pvEvent Handle to event being reset                               */
/*****************************************************************************/
void OS_ResetEvent(void* pvEvent)
{
  ResetEvent((HANDLE)pvEvent);
}

/*****************************************************************************/
/*! Wait for the signalling of an event
*   \param pvEvent   Handle to event being wait for
*   \param ulTimeout Timeout in ms to wait for event
*   \return 0 if event was signalled                                         */
/*****************************************************************************/
uint32_t OS_WaitEvent(void* pvEvent, uint32_t ulTimeout)
{
  return (WaitForSingleObject((HANDLE)pvEvent, ulTimeout) == WAIT_OBJECT_0) ? CIFX_EVENT_SIGNALLED : CIFX_EVENT_TIMEOUT;
}

/*****************************************************************************/
/*! Delete an event
*   \param pvEvent Handle to event being deleted                             */
/*****************************************************************************/
void OS_DeleteEvent(void* pvEvent)
{
  CloseHandle((HANDLE)pvEvent);
}


/*****************************************************************************/
/*! This function enables the interrupts for the device physically
*   \param pvOSDependent O/S Dependent Variable passed during call to
*                        cifXTKitAddDevice                                   */
/*****************************************************************************/
void OS_EnableInterrupts(void* pvOSDependent)
{
  PCIFXCEDRV_DEVICEINSTANCE ptDrvDevInst = (PCIFXCEDRV_DEVICEINSTANCE)pvOSDependent;
  BOOLEAN                   fSuccess     = FALSE;
  DWORD                     dwThreadID   = 0;

  ptDrvDevInst->fStopIst = FALSE;

  /* Check if ISR DLL and handler function are provided by the user */
  if ( (0 == _tcslen(ptDrvDevInst->tIsrInfo.szIsrDll))    &&
       (0 == _tcslen(ptDrvDevInst->tIsrInfo.szIsrHandler))   )
  {
    /* Use generic ISR DLL and handler function */
    _tcsncpy(ptDrvDevInst->tIsrInfo.szIsrDll, _T("giisr.dll"), DEVDLL_LEN);
    _tcsncpy(ptDrvDevInst->tIsrInfo.szIsrHandler, _T("ISRHandler"), DEVENTRY_LEN);
  }

  if ( NULL == (ptDrvDevInst->hIntChain = LoadIntChainHandler(ptDrvDevInst->tIsrInfo.szIsrDll,
                                                              ptDrvDevInst->tIsrInfo.szIsrHandler,
                                                              (BYTE)ptDrvDevInst->tIsrInfo.dwIrq)))
  {
    if(g_ulTraceLevel & TRACE_LEVEL_ERROR)
    {
      USER_Trace(&ptDrvDevInst->tDevInstance, 
                TRACE_LEVEL_ERROR,
                "Error loading installable ISR. Device will not be able to work in interrupt mode.\r\n");
    }

  } else
  {
    GIISR_INFO tGIISRInfo = {0};
    tGIISRInfo.SysIntr    = ptDrvDevInst->tIsrInfo.dwSysintr;
    tGIISRInfo.PortIsIO   = FALSE;
    tGIISRInfo.UseMaskReg = FALSE;  
    
    if (ptDrvDevInst->fPLXBridge)
    {
      tGIISRInfo.CheckPort = TRUE;

      /* Our installable interrupt service routine, must be given a static,
         process independent, virtual mapped address to the device it will access 
         in the interrupt context. As pbPLXRegisters is already mapped this way 
         we don't need to create another mapping */
      tGIISRInfo.PortAddr  = (DWORD)ptDrvDevInst->pbPLXRegisters+PLX_PCI_INTCTRLSTS_REG;

      tGIISRInfo.PortSize  = 1;
      tGIISRInfo.Mask      = PLX_HW_INTERRUPT_STATE;
    } else if (ptDrvDevInst->tDevInstance.fPCICard)
    {
      /* Setup pointer to netX interrupt state register */
      uint32_t ulIRQStatePhyAddr = (ptDrvDevInst->tDevInstance.ulPhysicalAddress>>8) + 
                                   ptDrvDevInst->tDevInstance.ulDPMSize -
                                   sizeof(NETX_GLOBAL_REG_BLOCK) +
                                   offsetof(NETX_GLOBAL_REG_BLOCK, ulIRQState_0);
      tGIISRInfo.CheckPort       = TRUE;

      /* Our installable interrupt service routine, must be given a static,
         process independent, virtual mapped address to the device it will access 
         in the interrupt context. So we need to create the address mapping here */
      tGIISRInfo.PortAddr        = (DWORD)CreateStaticMapping(ulIRQStatePhyAddr, 4);

      tGIISRInfo.PortSize        = 4;
      tGIISRInfo.Mask            = MSK_IRQ_STA0_INT_REQ;
    } else
    {
      /* This is an ISA card. As the interrupt line that the device is using 
         is not shared, the ISR handler does not need to determine whether the 
         device is asserting the IRQ */
      tGIISRInfo.CheckPort = FALSE;
      tGIISRInfo.PortAddr  = NULL;
      tGIISRInfo.PortSize  = 0;
      tGIISRInfo.Mask      = 0;
    }

    if (!KernelLibIoControl(ptDrvDevInst->hIntChain, IOCTL_GIISR_INFO,&tGIISRInfo,sizeof(tGIISRInfo),NULL,0,NULL))
    {
      if(g_ulTraceLevel & TRACE_LEVEL_ERROR)
      {
        USER_Trace(&ptDrvDevInst->tDevInstance, 
                   TRACE_LEVEL_ERROR,
                   "Error calling LibIOControl. Device will not be able to work in interrupt mode.\r\n");
      }

    } else if (NULL == (ptDrvDevInst->hIrqEvent = CreateEvent(NULL, FALSE, FALSE, NULL)))
    {
      if(g_ulTraceLevel & TRACE_LEVEL_ERROR)
      {
        USER_Trace(&ptDrvDevInst->tDevInstance, 
                   TRACE_LEVEL_ERROR, 
                   "Error creating interrupt event. Card will not work in this configuration!");
      }

    } else if(!InterruptInitialize(ptDrvDevInst->tIsrInfo.dwSysintr, 
                                   ptDrvDevInst->hIrqEvent,NULL,0))
    {
      if(g_ulTraceLevel & TRACE_LEVEL_ERROR)
      {
        USER_Trace(&ptDrvDevInst->tDevInstance, 
                   TRACE_LEVEL_ERROR, 
                   "Error initializing interrupts (%d). Card will not work in this configuration!",
                   GetLastError());
      }

    } else if(NULL == (ptDrvDevInst->hIst = CreateThread(NULL,0,IST,ptDrvDevInst,
                                                         0,&dwThreadID)))
    {
      if(g_ulTraceLevel & TRACE_LEVEL_ERROR)
      {
        USER_Trace(&ptDrvDevInst->tDevInstance, 
                   TRACE_LEVEL_ERROR, 
                   "Error creating interrupt thread (%d). Card will not work in this configuration!",
                   GetLastError());
      }
    } else
    {
      if(TRUE == ptDrvDevInst->fPLXBridge)
      {
        /* Enable PLX Interrupts  */
        BYTE bData = ptDrvDevInst->pbPLXRegisters[PLX_PCI_INTCTRLSTS_REG];
        bData |= PLX_HW_INTERRUPT_ENABLE;

        /* Write new data to port */
        ptDrvDevInst->pbPLXRegisters[PLX_PCI_INTCTRLSTS_REG] = bData;
      }
      InterruptDone(ptDrvDevInst->tIsrInfo.dwSysintr);
      fSuccess = TRUE;
    }
  }



  /* Cleanup if something fails */
  if(!fSuccess)
    OS_DisableInterrupts(ptDrvDevInst);
}


/*****************************************************************************/
/*! This function enables the interrupts for the device physically
*   \param pvOSDependent O/S Dependent Variable passed during call to
*                        cifXTKitAddDevice                                   */
/*****************************************************************************/
void OS_DisableInterrupts(void* pvOSDependent)
{
  PCIFXCEDRV_DEVICEINSTANCE ptDrvDevInst = (PCIFXCEDRV_DEVICEINSTANCE)pvOSDependent;
  
  if(ptDrvDevInst->hIst != NULL)
  {
    InterruptDisable(ptDrvDevInst->tIsrInfo.dwSysintr);
    ptDrvDevInst->fStopIst = TRUE;
    SetEvent(ptDrvDevInst->hIrqEvent);

    if(WaitForSingleObject(ptDrvDevInst->hIst, 1000) != WAIT_OBJECT_0)
      TerminateThread(ptDrvDevInst->hIst, MAXDWORD);

    CloseHandle(ptDrvDevInst->hIst);
    ptDrvDevInst->hIst = NULL;
  }
  
  if(ptDrvDevInst->hIrqEvent)
  {
    CloseHandle(ptDrvDevInst->hIrqEvent);
    ptDrvDevInst->hIrqEvent = NULL;
  }
    
  if (ptDrvDevInst->hIntChain)
    FreeIntChainHandler(ptDrvDevInst->hIntChain);
}